<?php

namespace App\Notifications;

use App\Enums\BookingItemStatus;
use App\Models\BookingItem;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use NotificationChannels\Fcm\FcmChannel;
use NotificationChannels\Fcm\FcmMessage;
use NotificationChannels\Fcm\Resources\Notification as FcmNotification;

class ItemStatusChangedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $item;
    protected $status; // 'confirmed' or 'declined' or 'completed'
    protected $user;

    public function __construct(BookingItem $item, BookingItemStatus $status, User $user)
    {
        $this->item = $item;
        $this->status = $status;
        $this->user = $user;
    }

    public function via($notifiable)
    {
        $channels = ['database'];

        if ($notifiable->firebase_token) {
            $channels[] = FcmChannel::class;
        }

        return $channels;
    }

    public function toFcm($notifiable): FcmMessage
    {
        $locale = $notifiable->pref_lang ?? app()->getLocale();
        $titleKey = match ($this->status) {
            BookingItemStatus::CONFIRMED => 'item_confirmed_title',
            BookingItemStatus::DECLINED => 'item_declined_title',
            BookingItemStatus::COMPLETED => 'item_completed_title',
            BookingItemStatus::CANCELED => 'item_cancelled_title',
            default => '',
        };
        $bodyKey = match ($this->status) {
            BookingItemStatus::CONFIRMED => 'item_confirmed_body',
            BookingItemStatus::DECLINED => 'item_declined_body',
            BookingItemStatus::COMPLETED => 'item_completed_body',
            BookingItemStatus::CANCELED => 'item_cancelled_body',
            default => '',
        };
        $title = __($titleKey, ['product' => $this->item->product->name ?? 'Product'], $locale);
        $body = __($bodyKey, ['item_id' => $this->item->id, 'status' => $this->status->value, 'date' => $this->item->service_date], $locale);

        return FcmMessage::create()
            ->data([
                'item_id' => (string) $this->item->id,
                'booking_id' => (string) $this->item->booking_id,
                'url' => route('user.bookings.show', $this->item->booking_id),
                'type' => 'item_status_changed',
            ])
            ->notification(
                notification: FcmNotification::create()
                    ->title($title)
                    ->body($body)
            );
    }

    public function toArray($notifiable)
    {
        $locale = $notifiable->pref_lang ?? app()->getLocale();
        $titleKey = match ($this->status) {
            BookingItemStatus::CONFIRMED => 'item_confirmed_title',
            BookingItemStatus::DECLINED => 'item_declined_title',
            BookingItemStatus::COMPLETED => 'item_completed_title',
            BookingItemStatus::CANCELED => 'item_cancelled_title',
            default => '',
        };
        $bodyKey = match ($this->status) {
            BookingItemStatus::CONFIRMED => 'item_confirmed_body',
            BookingItemStatus::DECLINED => 'item_declined_body',
            BookingItemStatus::COMPLETED => 'item_completed_body',
            BookingItemStatus::CANCELED => 'item_cancelled_body',
            default => '',
        };

        return [
            'title_en' => __($titleKey, ['product' => $this->item->product->name ?? 'Product'], 'en'),
            'body_en' => __($bodyKey, ['item_id' => $this->item->id, 'status' => $this->status->value, 'date' => $this->item->service_date], 'en'),
            'title_ar' => __($titleKey, ['product' => $this->item->product->name ?? 'Product'], 'ar'),
            'body_ar' => __($bodyKey, ['item_id' => $this->item->id, 'status' => $this->status->value, 'date' => $this->item->service_date], 'ar'),
            'url' => route('user.bookings.show', $this->item->booking_id),
            'booking_id' => $this->item->booking_id,
        ];
    }
}
