<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ContactResource\Pages;
use App\Filament\Resources\ContactResource\RelationManagers;
use App\Models\Contact;
use Filament\Forms;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Toggle; // Added for is_resolved
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn; // Added for is_resolved
use Filament\Tables\Filters\Filter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class ContactResource extends Resource
{
    protected static ?string $model = Contact::class;

    protected static ?string $navigationIcon = 'heroicon-o-chat-bubble-left-right';
    protected static ?string $navigationGroup = 'app.filament.navigation.group.support';
    protected static ?int $navigationSort = 1;

    public static function getLabel(): ?string
    {
        return __('app.filament.navigation.contact');
    }
    public static function getPluralLabel(): ?string
    {
        return __('app.filament.navigation.contacts');
    }

    public static function getNavigationSort(): int
    {
        return static::$navigationSort;
    }

    public static function getNavigationGroup(): string
    {
        return __('app.filament.navigation.group.support');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::$model::count();
    }

    public static function getNavigationBadgeColor(): ?string
    {
        $count = static::$model::where('is_resolved', false)->count();
        return $count > 0 ? 'danger' : 'success';
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('name')
                    ->label(__('app.filament.fields.name'))
                    ->disabled()
                    ->required()
                    ->maxLength(255),
                TextInput::make('email')
                    ->label(__('app.filament.fields.email'))
                    ->disabled()
                    ->email()
                    ->required()
                    ->maxLength(255),
                TextInput::make('subject')
                    ->label(__('app.filament.fields.subject'))
                    ->disabled()
                    ->required()
                    ->maxLength(255),
                Textarea::make('message')
                    ->label(__('app.filament.fields.message'))
                    ->required()
                    ->disabled()
                    ->columnSpanFull(),
                Textarea::make('notes')
                    ->label(__('app.filament.fields.admin_notes'))
                    ->columnSpanFull(),
                Toggle::make('is_resolved') // Added for is_resolved
                    ->label(__('app.filament.fields.is_resolved'))
                    ->inline(false),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('id')
                    ->label(__('ID'))
                    ->sortable(),
                TextColumn::make('name')
                    ->label(__('app.filament.fields.name'))
                    ->searchable(),
                TextColumn::make('email')
                    ->label(__('app.filament.fields.email'))
                    ->searchable(),
                TextColumn::make('subject')
                    ->label(__('app.filament.fields.subject'))
                    ->searchable(),
                IconColumn::make('is_resolved') // Added for is_resolved
                    ->label(__('app.filament.fields.is_resolved'))
                    ->boolean(),
                TextColumn::make('created_at')
                    ->label(__('app.filament.fields.created_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('updated_at')
                    ->label(__('app.filament.fields.updated_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Filter::make('created_at')
                    ->form([
                        Forms\Components\DatePicker::make('created_from')->label(__('Created From')),
                        Forms\Components\DatePicker::make('created_until')->label(__('Created To')),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['created_from'],
                                fn(Builder $query, $date): Builder => $query->whereDate('created_at', '>=', $date),
                            )
                            ->when(
                                $data['created_until'],
                                fn(Builder $query, $date): Builder => $query->whereDate('created_at', '<=', $date),
                            );
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListContacts::route('/'),
        ];
    }
}
